/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/* ***********************************************************************
   *                                                                     *
   * cpuport.c                                                           *
   * Christian Zurhorst, BID R&D, 3. Jan. 1998                           *
   *                                                                     *
   * CPU port functions                                                  *
   *                                                                     *
   ***********************************************************************
*/


#include <typedefs.h>

#include <cpuport.h>
#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regconst.h>
#include <regx1a.h>
#include <regx1b.h>
#include <regx1c.h>
#include <regx1d.h>
#include <regx6a.h>
#include <regx6b.h>
#include <regx6c.h>
#include <regx6d.h>
#include <session.h>

#include "b_cmd.h"


b_errtype EXPORT BestCPUportByteBlockWrite( /* @cpubbwrite */
    b_handletype handle,
    b_int32 device_num,         /* @dev  */
    b_int32 address,            /* @ad */
    b_int8ptr data_ptr,         /* @data */
    b_int32 num_of_bytes        /* @nob */
);

b_errtype EXPORT BestCPUportByteBlockRead(  /* @cpubbread */
    b_handletype handle,
    b_int32 device_num,         /* @dev */
    b_int32 address,            /* @ad */
    b_int8ptr data_ptr,         /* #RETURN "data =: %s\n" */
    b_int32 num_of_bytes        /* @nob */
);



/* --------------------------------------------------------------------------
 * Centralized checking of all params for CPU Port I/O single mode
 * -------------------------------------------------------------------------- */

static b_errtype BestCpuParamCheck(
    b_handletype handle,
    b_ccharptrtype pFuncName,
    b_int32 device_num,
    b_int32 address,
    b_int32ptr data_ptr,
    b_sizetype size)

{
  B_TRY_VARS_NO_PROG;
  B_TRY_BEGIN
  {
    B_TRY_FAIL (BestIsCompact(handle) ? B_E_NOT_COMPACT : B_E_OK);
    /* function name is passed as argument */
    B_TRY_FCT_PARAM(device_num, device_num > 1);
    B_TRY_FCT_PARAM_RANGE(size, B_SIZE_BYTE, B_SIZE_WORD);
    B_TRY_FCT_PARAM(address, address > CPUPORT_MAX_DATA_OFFSET);
    B_TRY_FCT_PARAM_ALIGNMENT(address, size);
    B_TRY_FCT_PARAM_NULL_POINTER(data_ptr);
  }

  return B_TRY_RET;
}
/* --------------------------------------------------------------------------
 * Centralized checking of all params for CPU Port I/O Block mode
 * -------------------------------------------------------------------------- */

static b_errtype BestCpuBlockParamCheck(
    b_handletype handle,
    b_ccharptrtype pFuncName,
    b_int32 device_num,
    b_int32 address,
    b_int8ptr data_ptr,
    b_sizetype size,
    b_int32 num_of_bytes)

{
  B_TRY_VARS_NO_PROG;
  B_TRY_BEGIN
  {
    B_TRY_FAIL (BestIsCompact(handle) ? B_E_NOT_COMPACT : B_E_OK);
    B_TRY_FCT_PARAM(device_num, device_num > 1);
    B_TRY_FCT_PARAM_ALIGNMENT(address, size);
    B_TRY_FCT_PARAM(address, address > CPUPORT_MAX_DATA_OFFSET);
    B_TRY_FCT_PARAM_NULL_POINTER(data_ptr);
    B_TRY_FCT_PARAM_R(num_of_bytes, (address + num_of_bytes) > CPUPORT_MAX_DATA_SIZE,
      "Block exceeds port boundary.");

    /* CONSIDER: code should handle this without error! CZ */
    B_TRY_FCT_PARAM(num_of_bytes, num_of_bytes == 0);

    B_TRY_FCT_PARAM_ALIGNMENT(num_of_bytes, size);
  }

  return B_TRY_RET;
}
/* --------------------------------------------------------------------------
 * Helper function for setting params
 * ------------------------------------------------------------------------- */
static b_errtype BestCpuPortParamSet(
    b_handletype handle,
    b_int16 address,
    b_int8 device_num,
    b_int8 size)
{
  b_int8 cmdbuf[4];
  b_int8ptr p_cmdbuf = cmdbuf;
  p_cmdbuf = BestWord2Stream(p_cmdbuf, &address, 1UL);
  p_cmdbuf = BestByteCopy(p_cmdbuf, &device_num, 1UL);
  (void) BestByteCopy(p_cmdbuf, &size, 1UL);

  return BestBasicCommand(handle, CMD_CPUPORT_PARAM_SET,
    cmdbuf, IN_CPUPORT_PARAM_SET, NULL, NULL);
}


/* --------------------------------------------------------------------------
 * Set the default for the four cpu port properties
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestCPUportPropDefaultSet(
    b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i;
  b_param_infotype *ParamInfo;
  b_generic_infotype *GenInfo;
  /* license checking */
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    B_TRY(BestGenInfoGet(handle, B_PARAM_CPUPORT, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_CPUPORT, i, &ParamInfo,
          (b_int32) B_INDEX_SEARCH));

      /* Set it to default */
      B_TRY(BestCPUportPropSet(handle, ParamInfo->proptyp.cpuprop,
          ParamInfo->defaultval));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Set the CPU port properties. (uses for E2926 and E2925 the dynamic c.)
 * ------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportPropSet(
    b_handletype handle,
    b_cpuproptype cpuprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestCPUportPropSet [cpuprpset]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestParamCheck(handle, B_PARAM_CPUPORT, (b_int32) cpuprop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropByteSet(handle,
          CMD_CPUPORT_PROP_SET, (b_int8) cpuprop, value));
    }

    else
    {
      b_int8 val;
      /* out of range errors are already checked by dynamic capabilities */
      B_TRY_FCT_PARAM(3, value > 0xff);
      val = (b_int8) value;

      switch (cpuprop)
      {
      case B_CPU_MODE:
        B_TRY_FCT_PARAM(value, ((value != B_CM_MASTER) && (value != B_CM_DISABLED)));
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_MODE, &val, 1UL));
        break;

      case B_CPU_PROTOCOL:
        B_TRY_FCT_PARAM(value, value != B_CP_INTEL);
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_PROTOCOL, &val, 1UL));
        break;

      case B_CPU_RDYTYPE:
        B_TRY_FCT_PARAM(value, ((value != B_CR_EXTERNAL) && (value != B_CR_AUTO)));
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_RDYTYPE, &val, 1UL));
        break;

      default:
        B_FCT_PARAM_ERROR(cpuprop, "Unknown cpu prop. type");
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportWrite(
    b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int32 data,
    b_sizetype size
)
{
  B_DECLARE_FUNCNAME("BestCPUportWrite [cpuwrite]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuParamCheck(handle, pFuncName, device_num, address,
        &data, size));

    if (Best16BitRegisterFile(handle))
    {
      b_int8 buf[2];
      if (size == B_SIZE_BYTE)
      {
        b_int8 data8 = (b_int8) data;
        B_TRY_FCT_PARAM(data, data > 0xff);

        (void) BestByteCopy(buf, &data8, 1UL);
      }
      else
      {
        b_int16 data16 = (b_int16) data;
        B_TRY_FCT_PARAM(data, data > 0xffff);

        (void) BestWord2Stream(buf, &data16, 1UL);
      }

      /* set params */
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));


      /* do the write */
      B_TRY(BestBasicCommandVariable(handle, CMD_CPUPORT_WRITE,
          buf, size, NULL, NULL));
    }

    else
    {
      b_int32 value = (b_int32) size;
      /* write size, address, data and start transfer with data writing */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & value, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DATA, (b_int8ptr) & data, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportRead(
    b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int32 * data_ptr,
    b_sizetype size
)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuParamCheck(handle, "BestCPUportRead", device_num, address,
        data_ptr, size));

    if (Best16BitRegisterFile(handle))
    {
      b_int8 cmdbuf[4];
      b_int16 length = size;
      b_int16 outsize = OUT_CPUPORT_READ;
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));

      (void) BestWord2Stream(cmdbuf, &length, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_CPUPORT_READ,
          cmdbuf, IN_CPUPORT_READ, cmdbuf, &outsize));

      if (size == B_SIZE_BYTE)
      {
        *data_ptr = cmdbuf[0];
      }
      else
      {
        b_int16 wdata;
        (void) BestStream2Word(&wdata, cmdbuf, 1UL);
        *data_ptr = wdata;
      }
    }

    else
    {
      b_int32 value = (b_int32) size;
      /* write size, address and read data */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & value, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));
      B_TRY(BestBasicBlockRead(handle, CPU_PORT_DATA, (b_int8ptr) data_ptr, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * SCR; 12/97; CPU Port Block I/O ... Pay Attention !!
 * CPU Block I/O is compatible with the 2925 (new firmware version).
 * For a 2926 BestCPUportBlockRead()/BestCPUportBlockWrite() will prepare
 * the Best and then use the generic block I/O function...with ONE exception;
 * For a Fast-Host-Interface with a desired size of 8-bit we use the
 * CPU-controlled method because the FHIF access on the 2926 requires
 * 16-bit granularity.
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 *  * NOTE; Not actually exposed yet!
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportByteBlockWrite(b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int8ptr data_ptr,
    b_int32 num_of_bytes)
{
  B_DECLARE_FUNCNAME("BestCPUportByteBlockWrite [cpubbwrite]");

  B_TRY_VARS_NO_PROG;
  b_int8 size = B_SIZE_BYTE;
  b_int8ptr p_data8 = data_ptr;
  b_int32 data;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuBlockParamCheck(handle, pFuncName, device_num,
        address, data_ptr, size, num_of_bytes));

    if (Best16BitRegisterFile(handle))
    {
      /* setup params */
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));


      /* size is byte therefore use single byte accesses in firmware */

      /* do the write */
      B_TRY(BestBasicCommandVariable(handle, CMD_CPUPORT_WRITE,
          data_ptr, (b_int16) num_of_bytes, NULL, NULL));

    }

    else
    {
      /* write size, address, data and start transfer with data writing */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & size, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));

      for (; num_of_bytes; num_of_bytes -= size, p_data8 += size)
      {
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
        /* write data in a loop ... Best increments addr. by CPU_PORT_SIZE
         * after each read/write */
        data = (b_int32) * p_data8;
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DATA,
            (b_int8ptr) & data, 4UL));
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * NOTE; Not actually exposed yet!
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportByteBlockRead(b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int8ptr data_ptr,
    b_int32 num_of_bytes)
{
  B_DECLARE_FUNCNAME("BestCPUportByteBlockRead [cpubbread]");

  B_TRY_VARS_NO_PROG;
  b_int8 size = B_SIZE_BYTE;
  b_int8ptr p_data8 = data_ptr;
  b_int16 data;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuBlockParamCheck(handle, pFuncName, device_num,
        address, data_ptr, size, num_of_bytes));

    if (Best16BitRegisterFile(handle))
    {
      b_int16 outsize = OUT_CPUPORT_READ;
      b_int16 length = (b_int16) num_of_bytes;
      b_int8 cmdbuf[2];
      /* must setup parameters before reading */
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));


      (void) BestWord2Stream(cmdbuf, &length, 1UL);

      B_TRY(BestBasicCommand(handle, CMD_CPUPORT_READ,
          cmdbuf, IN_CPUPORT_READ, p_data8, &outsize));

    }
    else
    {
      /* write size, address, data and start transfer with data writing */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & size, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));

      for (; num_of_bytes; num_of_bytes -= size, p_data8 += size)
      {
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
        /* write data in a loop ... */
        B_TRY(BestBasicBlockRead(handle, CPU_PORT_DATA, (b_int8ptr) & data, 4UL));

        *p_data8 = (b_int8) data;

      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* ---------------------------------------------------------------------------
 *
 * ---------------------------------------------------------------------------*/

b_errtype EXPORT BestCPUportWordBlockWrite(b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int16ptr data_ptr,
    b_int32 num_of_bytes)
{
  B_DECLARE_FUNCNAME("BestCPUportWordBlockWrite [cpuwbwrite]");

  B_TRY_VARS;
  b_int8 size = B_SIZE_WORD;
  b_int32 data;
  b_blockiorestype resource;    /* onboard resource to read/write */

  /* variables used after a catch must be declared volatile */
  volatile b_int8ptr p_data8 = (b_int8ptr) data_ptr;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuBlockParamCheck(handle, pFuncName, device_num,
        address, (b_int8ptr) data_ptr, size, num_of_bytes));

#if 0
  b_porttype port;
    B_TRY(BestGetPortFromHandle(handle, &port));

    if (Best16BitRegisterFile(handle) && (port != B_PORT_RS232))

#else
    if (Best16BitRegisterFile(handle))

#endif
    {
      /* setup params */
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));
      p_data8 = (b_int8ptr) malloc((size_t) num_of_bytes);
      B_TRY_PROGRESS((p_data8 == NULL) ? B_E_HOST_MEM_FULL : B_E_OK);

      /* copy data buffer to new buffer and swap the byte order */
      (void) BestWord2Stream(p_data8, data_ptr, (num_of_bytes >> 1));

      /* size is word -> we can use the fast block mode */

      if (device_num == 0)
      {
        resource = B_BIO_CPU_PORT0;
      }

      else
      {
        resource = B_BIO_CPU_PORT1;
      }

      /* the actual write */
      B_TRY(BestBasicBlockCommand(handle, resource, address,
          num_of_bytes, (b_int32) size, CMD_WRITE, p_data8));
    }

    else
    {
      /* write size, address, data and start transfer with data writing */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & size, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));

      for (; num_of_bytes; num_of_bytes -= size, p_data8 += size)
      {
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
        /* write data in a loop ... Best increments addr. by CPU_PORT_SIZE
         * after each read/write */
        data = (b_int32) * ((b_int16 *) p_data8);
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DATA,
            (b_int8ptr) & data, 4UL));
      }
    }

    B_TRY_ERROR(B_E_INTERNAL_RETURN);
  }

  B_TRY_CATCH
  {
    B_TRY_PASSED
    {
      free(p_data8);

      if (B_TRY_RET == B_E_INTERNAL_RETURN)
        B_TRY_RET = B_E_OK;
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportWordBlockRead(b_handletype handle,
    b_int32 device_num,
    b_int32 address,
    b_int16ptr data_ptr,
    b_int32 num_of_bytes)
{
  B_DECLARE_FUNCNAME("BestCPUportWordBlockRead [cpuwbread]");

  B_TRY_VARS;
  b_int8 size = B_SIZE_WORD;
  b_int16 data;
  b_blockiorestype resource;    /* onboard resource to read/write */

  /* variables used after a catch must be declared volatile */
  volatile b_int8ptr p_data8 = (b_int8ptr) data_ptr;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestCpuBlockParamCheck(handle, pFuncName, device_num,
        address, (b_int8ptr) data_ptr, size, num_of_bytes));

#if 0
  b_porttype port;
    B_TRY(BestGetPortFromHandle(handle, &port));

    if (Best16BitRegisterFile(handle) && (port != B_PORT_RS232))

#else
    if (Best16BitRegisterFile(handle))

#endif
      
    {

      /* must setup parameters before reading */
      B_TRY(BestCpuPortParamSet(handle, (b_int16) address,
          (b_int8) device_num, (b_int8) size));

      p_data8 = (b_int8ptr) malloc((size_t) num_of_bytes);
      B_TRY_PROGRESS((p_data8 == NULL) ? B_E_HOST_MEM_FULL : B_E_OK);

      if (device_num == 0)
      {
        resource = B_BIO_CPU_PORT0;
      }
      else
      {
        resource = B_BIO_CPU_PORT1;
      }
      /* the actual read */
      B_TRY(BestBasicBlockCommand(handle, resource, address,
          num_of_bytes, (b_int32) size, CMD_READ, p_data8));
      (void) BestStream2Word(data_ptr, p_data8, (num_of_bytes >> 1));
    }

    else
    {
      /* write size, address, data and start transfer with data writing */
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_SIZE, (b_int8ptr) & size, 1UL));
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_DEVICE, (b_int8ptr) & device_num, 1UL));

      for (; num_of_bytes; num_of_bytes -= size, p_data8 += size)
      {
        B_TRY(BestBasicBlockWrite(handle, CPU_PORT_ADDR, (b_int8ptr) & address, 2UL));
        /* write data in a loop ... */
        B_TRY(BestBasicBlockRead(handle, CPU_PORT_DATA, (b_int8ptr) & data, 4UL));
        *((b_int16 *) p_data8) = data;

      }
    }

    B_TRY_ERROR(B_E_INTERNAL_RETURN);
  }

  B_TRY_CATCH
  {
    B_TRY_PASSED
    {
      free(p_data8);

      if (B_TRY_RET == B_E_INTERNAL_RETURN)
        B_TRY_RET = B_E_OK;
    }
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * Retrieve the status of Cpu port INT# line.
 * Note; Even though INT# is active low this functions returns True if
 * there is an interrupt pending.
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportIntrStatusGet(
    b_handletype handle,
    b_int32 * intvalue_ptr
)
{
  B_TRY_VARS_NO_PROG;
  b_int8 value;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BBCReceiveByte(handle, CMD_CPUPORT_INTR_GET, &value));
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, CPU_PORT_INTR, &value, 1UL));
    }

    *intvalue_ptr = value ? 1 : 0;
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * This function clears the CPUport INT# Latch
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportIntrClear(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_CPUPORT_INTR_CLR, NULL, 0, NULL, NULL));
    }
    else
    {
      b_int8 value = 1;
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_INTR, &value, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * Set the CPU port RST# line Active (Low; value==0) or Inactive (High; value!=0)
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestCPUportRST(
    b_handletype handle,
    b_int32 value
)
{
  B_TRY_VARS_NO_PROG;
  b_int8 rst = (b_int8) (value ? 1 : 0);
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BBCSendByte(handle, CMD_CPUPORT_RESET, &rst));
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, CPU_PORT_RESET, &rst, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}
/* -----------------------------------------------------------------
 * Static IO Port Programming Functions
 * ----------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * Sets the mode for one pin on the Static I/O port.
 * value: 0 = Input, 1 = Totem Pole Output, 2 = Open Drain Output
 * staticprop: Must be B_STAT_PINMODE (no other params supported).
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Set the default for the static io properties
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestStaticPropDefaultSet(
    b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i, pin;
  b_param_infotype *ParamInfo;
  b_generic_infotype *GenInfo;
  /* license checking */
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    B_TRY(BestGenInfoGet(handle, B_PARAM_STATICIO, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_STATICIO, i, &ParamInfo,
          (b_int32) B_INDEX_SEARCH));

      /* Set it to default */
      for (pin = 0; pin < GenInfo->depth; pin++)
        B_TRY(BestStaticPropSet(handle, pin, ParamInfo->proptyp.staticprop,
            ParamInfo->defaultval));
    }
  }

  B_ERRETURN(B_TRY_RET);
}
/* --------------------------------------------------------------------------
 * Set the the static io properties
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestStaticPropSet(
    b_handletype handle,
    b_int32 pin_num,
    b_staticproptype staticprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestStaticPropSet [sprpset]");
  B_TRY_VARS_NO_PROG;
  b_int16 regval = 0;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY_FCT_PARAM(1, pin_num > 7);
    B_TRY(BestParamCheck(handle, B_PARAM_STATICIO, (b_int32) staticprop, value));

    /* read current configuration */
    if (Best16BitRegisterFile(handle))
    {
      /* cheat a bit, static IO pins only have one property, but all of them
       * have the same. So send pin number instead of property identifier. */
      B_TRY(BestAbstractPropLongSet(handle, CMD_STATIC_IO_PROP_SET,
          (b_int8) pin_num, value));
    }
    else
    {
      B_TRY(BestBasicBlockRead(handle, SIDEBAND_CONF, (b_int8ptr) & regval, 4UL));

      /* zero out the 2 bits that describe the pin's configuration (only) */
      regval &= (b_int16) (~(0x03UL << (int) (pin_num * 2)));

      /* set the bits required to describe the new config. */
      switch (value)
      {
      case B_PMD_INPONLY:
        break;
      case B_PMD_TOTEMPOLE:
        regval |= (b_int16) (1 << (b_int16) (pin_num * 2));
        break;
      case B_PMD_OPENDRAIN:
        regval |= (b_int16) (2 << (b_int16) (pin_num * 2));
        break;
      default:
        B_TRY_ERROR(B_E_PARAM);
        break;
      }

      B_TRY(BestBasicBlockWrite(handle, SIDEBAND_CONF, (b_int8ptr) & regval, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

/* --------------------------------------------------------------------------
 * Set the the static io properties
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestStaticPropGet(
    b_handletype handle,
    b_int32 pin_num,
    b_staticproptype staticprop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestStaticPropGet [sprpget]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY_FCT_PARAM(1, pin_num > 7);
    if (!BestIsCompact(handle) || staticprop != B_STAT_PINMODE)
    {
      B_TRY(BestParamProbe(handle, B_PARAM_STATICIO, (b_int32) staticprop));
    }

    /* read current configuration */
    if (Best16BitRegisterFile(handle))
    {
      /* cheat a bit, static IO pins only have one property, but all of them
       * have the same. So send pin number instead of property identifier. */
      B_TRY(BestAbstractPropLongGet(handle, CMD_STATIC_IO_PROP_GET,
          (b_int8) pin_num, value));
    }
  }
  
  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Write one byte to the static I/O port
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestStaticWrite(
    b_handletype handle,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestStaticWrite [swrite]");

  b_int8 val = (b_int8) value;
  B_FCT_PARAM_CHK(1, value > 0xff);

  if (!BestIsCompact(handle))
  {
    if (Best16BitRegisterFile(handle))
    {
      B_ERRETURN(BBCSendByte(handle, CMD_STATIC_IO_WRITE, &val));
    }
    else
    {
      B_ERRETURN(BestBasicBlockWrite(handle, SIDEBAND_OUTPUT, &val, 1UL));
    }
  }
  else
  {
    B_ERRETURN (B_E_NOT_COMPACT);
  }
}


/* --------------------------------------------------------------------------
 * Read one byte from the static I/O port
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestStaticRead(
    b_handletype handle,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestStaticRead [sread]");

  b_errtype err;
  b_int8 val;
  B_FCT_PARAM_NULL_POINTER_CHK(value);

  if (!BestIsCompact(handle))
  {
    if (Best16BitRegisterFile(handle))
    {
      err = BBCReceiveByte(handle, CMD_STATIC_IO_READ, &val);
    }
    else
    {
      err = BestBasicBlockRead(handle, SIDEBAND_INPUT, &val, 1UL);
    }

    *value = val;
  }
  else
  {
    B_ERRETURN (B_E_NOT_COMPACT);
  }
  
  B_ERRETURN(err);
}


/* --------------------------------------------------------------------------
 * Sets the logical value of one single Static I/O pin.  All others stay
 * the same.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestStaticPinWrite(
    b_handletype handle,
    b_int32 pin_num,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestStaticPinWrite [spwrite]");

  b_int8 bitmask;
  b_errtype err;
  B_FCT_PARAM_CHK(1, pin_num > 7);
  B_FCT_PARAM_CHK(2, value > 0x02); /* 2 = toggle bit for 130 ms */

  if (!BestIsCompact(handle))
  {
    if (Best16BitRegisterFile(handle))
    {
      b_int8 dummy_8;
      b_int8 cmdbuf[2];
      b_int8ptr p_cmdbuf = cmdbuf;
      dummy_8 = (b_int8) pin_num;
      p_cmdbuf = BestByteCopy(p_cmdbuf, &dummy_8, 1UL);
      dummy_8 = (b_int8) value;
      (void) BestByteCopy(p_cmdbuf, &dummy_8, 1UL);

      B_ERRETURN(BestBasicCommand(handle, CMD_STATIC_IO_PIN_WR,
				  cmdbuf, IN_STATIC_IO_PIN_WR, NULL, NULL));
    }
    else
    {
      if (value == 0x2)           /* toggle a bit */
      {
	bitmask = (b_int8) (0x01UL << (b_int16) pin_num);
	B_ERRETURN(BestBasicBlockWrite(handle, SIDEBAND_INPUT, &bitmask, 1UL));
      }
      else
      {                           /* logical values 1 or 0 */
	/* read actual sideband output */
	B_ERRCHECK(BestBasicBlockRead(handle, SIDEBAND_OUTPUT, &bitmask, 1UL));

	/* modify output */
	if (value == 0)
	{
	  bitmask &= (b_int8) ~ (0x01UL << (b_int16) pin_num);
	}
	else
	{
	  bitmask |= (b_int8) (0x01UL << (b_int16) pin_num);
	}

	/* write back output */
	B_ERRETURN(BestBasicBlockWrite(handle, SIDEBAND_OUTPUT, &bitmask, 1UL));
      }
    }
  }
  else
  {
    B_ERRETURN (B_E_NOT_COMPACT);
  }

  B_ERRETURN (B_E_OK);
}
